<?php
namespace Piggly\Tests\Pix;

use Exception;
use PHPUnit\Framework\TestCase;
use Piggly\Pix\Parser;

/**
 * @coversDefaultClass \Piggly\Pix\Parser
 */
class ParserTest extends TestCase
{
	/**
	 * Assert if $document is validated as $expected.
	 *
	 * @covers ::parseDocument
	 * @covers ::validateDocument
	 * @dataProvider dataDocument
	 * @test Expecting positive assertion.
	 * @param boolean $expected Expected result.
	 * @param boolean $document Document to validate
	 * @return boolean
	 */
	public function canValidateDocument ( bool $expected, string $document )
	{ $this->assertEquals($expected, Parser::validateDocument($document)); }

	/**
	 * A list with random documents to validate.
	 * Provider to canValidateDocument() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataDocument () : array
	{
		return [
			// Must match dataset
			// CNPJ
			[true, '95.296.300/0001-76'],
			[true, '83.102.003/0001-63'],
			[true, '63.211.824/0001-20'],
			[true, '45.909.896/0001-64'],
			[true, '24.042.402/0001-17'],
			[true, '11.122.349/0001-12'],
			[true, '20.022.831/0001-71'],
			[true, '24.759.567/0001-04'],
			[true, '51.445.614/0001-18'],
			[true, '35.676.365/0001-15'],
			[true, '01.135.736/0001-30'],
			[true, '51.725.237/0001-70'],
			[true, '88.544.984/0001-12'],
			[true, '16.732.953/0001-84'],
			[true, '80.653.208/0001-49'],
			[true, '66.944.103/0001-18'],
			[true, '36.847.779/0001-22'],
			[true, '36.796.005/0001-10'],
			[true, '89.321.776/0001-17'],
			[true, '82.255.841/0001-04'],
			[true, '37.546.885/0001-39'],
			[true, '19.608.781/0001-20'],
			[true, '31.762.931/0001-50'],
			[true, '86.874.305/0001-48'],
			[true, '20.710.641/0001-47'],
			[true, '83.471.432/0001-08'],
			[true, '59.865.496/0001-09'],
			[true, '43.618.723/0001-61'],
			[true, '24.650.586/0001-06'],
			[true, '33.073.207/0001-45'],
			[true, '11.714.116/0001-09'],
			[true, '11.378.145/0001-47'],
			[true, '27.053.817/0001-84'],
			[true, '55.401.637/0001-81'],
			[true, '78.626.310/0001-03'],
			[true, '48.413.495/0001-07'],
			[true, '44.142.550/0001-10'],
			[true, '24.059.631/0001-44'],
			[true, '74.839.760/0001-23'],
			[true, '55.347.465/0001-05'],
			[true, '14.341.458/0001-19'],
			[true, '77.421.952/0001-03'],
			[true, '54.377.367/0001-58'],
			[true, '23.008.361/0001-80'],
			[true, '21.605.529/0001-09'],
			[true, '27.859.255/0001-60'],
			[true, '48.856.145/0001-15'],
			[true, '32.138.063/0001-03'],
			[true, '63.948.910/0001-10'],
			[true, '88.372.562/0001-07'],
			[true, '24139494000158'],
			[true, '24096722000150'],
			[true, '50841801000158'],
			[true, '04076657000193'],
			[true, '96355994000138'],
			[true, '42723798000140'],
			[true, '24194689000109'],
			[true, '16465135000162'],
			[true, '61335108000120'],
			[true, '91901728000140'],
			[true, '98367535000108'],
			[true, '72869196000166'],
			[true, '93315782000139'],
			[true, '96541444000103'],
			[true, '54018331000188'],
			[true, '24045989000118'],
			[true, '03216297000115'],
			[true, '95571489000168'],
			[true, '31756686000179'],
			[true, '53705402000158'],
			[true, '58653788000115'],
			[true, '37936370000145'],
			[true, '72164703000166'],
			[true, '24736758000150'],
			[true, '02796201000172'],
			[true, '36743277000151'],
			[true, '22258702000102'],
			[true, '63623009000179'],
			[true, '89139644000179'],
			[true, '98561480000164'],
			[true, '06974265000177'],
			[true, '74295402000105'],
			[true, '67885120000194'],
			[true, '88388627000102'],
			[true, '08353235000179'],
			[true, '32663927000106'],
			[true, '81865641000100'],
			[true, '83202546000152'],
			[true, '74146740000177'],
			[true, '36066715000194'],
			[true, '47486093000170'],
			[true, '28506937000152'],
			[true, '31886603000166'],
			[true, '98722829000100'],
			[true, '20571886000130'],
			[true, '87915813000190'],
			[true, '18086042000152'],
			[true, '03792934000100'],
			[true, '81020033000103'],
			[true, '24801422000124'],
			[true, '78.539.370/000199'],
			[true, '47.201.958/000104'],
			[true, '62.535174/0001-06'],
			[true, '03.919.7760001-06'],
			[true, '28.955.581/000134'],
			[true, '78.412.204/000127'],
			[true, '16.056.287/000101'],
			[true, '16.503047/0001-08'],
			[true, '43.775.1570001-00'],
			[true, '54.279548/0001-41'],
			[true, '63.885.594/000185'],
			[true, '47.291.0850001-78'],
			[true, '54.022341/0001-97'],
			[true, '94.306.0320001-63'],
			[true, '64.411.433/000112'],
			[true, '75829.999/0001-85'],
			[true, '86.443.5790001-82'],
			[true, '33.573541/0001-68'],
			[true, '47.587134/0001-14'],
			[true, '65.941.010/000177'],
			[true, '83.028653/0001-06'],
			[true, '62.432536/0001-33'],
			[true, '22.509.0870001-60'],
			[true, '54.924789/0001-04'],
			[true, '74.445.7400001-78'],
			[true, '88.605283/0001-46'],
			[true, '86955.855/0001-91'],
			[true, '72373.125/0001-78'],
			[true, '38.458.8150001-91'],
			[true, '59.409712/0001-01'],
			[true, '29826.713/0001-90'],
			[true, '31.822.010/000136'],
			[true, '52.099788/0001-39'],
			[true, '07.189.3190001-56'],
			[true, '44.739.196/000106'],
			[true, '13.993.367/000103'],
			[true, '66125.995/0001-25'],
			[true, '03093.115/0001-66'],
			[true, '63127.640/0001-87'],
			[true, '86.084205/0001-18'],
			[true, '63.845.941/000146'],
			[true, '26.124.9850001-50'],
			[true, '40.635369/0001-95'],
			[true, '75.636742/0001-07'],
			[true, '24.465.780/000103'],
			[true, '10.977.118/000128'],
			[true, '46.077.000/000190'],
			[true, '89693.767/0001-57'],
			[true, '18.293783/0001-04'],
			[true, '82663.954/0001-49'],
			// CPF
			[true, '354.060.756-08'],
			[true, '692.698.149-39'],
			[true, '411.811.468-28'],
			[true, '982.836.417-43'],
			[true, '897.433.091-10'],
			[true, '481.770.669-43'],
			[true, '702.491.180-84'],
			[true, '405.226.979-92'],
			[true, '179.140.675-04'],
			[true, '497.858.324-14'],
			[true, '644.076.930-88'],
			[true, '137.774.053-61'],
			[true, '456.182.249-68'],
			[true, '457.497.562-89'],
			[true, '181.290.021-08'],
			[true, '081.420.385-01'],
			[true, '540.261.628-00'],
			[true, '589.590.155-71'],
			[true, '850.408.524-44'],
			[true, '784.318.713-85'],
			[true, '619.783.787-00'],
			[true, '725.698.631-90'],
			[true, '960.299.051-12'],
			[true, '181.991.378-35'],
			[true, '898.819.063-77'],
			[true, '567.398.152-08'],
			[true, '469.227.968-02'],
			[true, '185.995.694-73'],
			[true, '940.792.772-53'],
			[true, '159.481.027-34'],
			[true, '267.029.142-29'],
			[true, '896.475.701-71'],
			[true, '608.596.180-48'],
			[true, '245.252.395-06'],
			[true, '315.949.657-06'],
			[true, '163.168.476-02'],
			[true, '191.760.188-30'],
			[true, '934.688.826-10'],
			[true, '807.420.391-37'],
			[true, '221.697.983-07'],
			[true, '083.355.975-37'],
			[true, '274.311.334-05'],
			[true, '549.981.232-20'],
			[true, '562.920.462-90'],
			[true, '791.963.313-30'],
			[true, '893.860.566-31'],
			[true, '464.592.775-24'],
			[true, '950.204.393-63'],
			[true, '949.468.143-61'],
			[true, '295.729.181-92'],
			[true, '41779611161'],
			[true, '18428741018'],
			[true, '22450235795'],
			[true, '97599698663'],
			[true, '96510995778'],
			[true, '01379815460'],
			[true, '70517970341'],
			[true, '01426299796'],
			[true, '28077322085'],
			[true, '90647370530'],
			[true, '02446337384'],
			[true, '11559270799'],
			[true, '25618063328'],
			[true, '50921914490'],
			[true, '45005833510'],
			[true, '98185433860'],
			[true, '47266140960'],
			[true, '29243396340'],
			[true, '41945071770'],
			[true, '66196594974'],
			[true, '53599028761'],
			[true, '56254718510'],
			[true, '72678990486'],
			[true, '44746528438'],
			[true, '07017770609'],
			[true, '61698356560'],
			[true, '94075672816'],
			[true, '95197457694'],
			[true, '54439037801'],
			[true, '54881842609'],
			[true, '19617289075'],
			[true, '17086868868'],
			[true, '61537125338'],
			[true, '27028276244'],
			[true, '79946228874'],
			[true, '52672776229'],
			[true, '42036408141'],
			[true, '05556414220'],
			[true, '20423504860'],
			[true, '14675584942'],
			[true, '99793618590'],
			[true, '49354266754'],
			[true, '54081519803'],
			[true, '29109848932'],
			[true, '35356649347'],
			[true, '62791297219'],
			[true, '00282968261'],
			[true, '09831374550'],
			[true, '48048109603'],
			[true, '28707938713'],
			[true, '250.427433-59'],
			[true, '650.771239-69'],
			[true, '353.553782-77'],
			[true, '683.714474-00'],
			[true, '611.530792-91'],
			[true, '715.980504-60'],
			[true, '543336.002-43'],
			[true, '002215.516-38'],
			[true, '801.941494-04'],
			[true, '733.761096-30'],
			[true, '260231.416-17'],
			[true, '291222.814-01'],
			[true, '976.902.06999'],
			[true, '422.892.04706'],
			[true, '667.089.03970'],
			[true, '373.773754-18'],
			[true, '280.113.13266'],
			[true, '503595.969-40'],
			[true, '085293.127-16'],
			[true, '399973.153-24'],
			[true, '526.588727-04'],
			[true, '953.449.14113'],
			[true, '924.165425-24'],
			[true, '754.998.64608'],
			[true, '522.470191-02'],
			[true, '420.802.79327'],
			[true, '309.142200-39'],
			[true, '098.687473-68'],
			[true, '194.530717-06'],
			[true, '816877.122-20'],
			[true, '081.334.78820'],
			[true, '579.194437-33'],
			[true, '117746.296-66'],
			[true, '228169.577-80'],
			[true, '803667.218-95'],
			[true, '770.911769-44'],
			[true, '980.844011-81'],
			[true, '181.653.25922'],
			[true, '627459.077-35'],
			[true, '393680.313-71'],
			[true, '635.358832-10'],
			[true, '445217.316-00'],
			[true, '783492.954-22'],
			[true, '938927.183-50'],
			[true, '299630.526-49'],
			[true, '833.052883-96'],
			[true, '022.981.44923'],
			[true, '385.704564-76'],
			[true, '240879.236-30'],
			[true, '372.353193-80'],
			// Must not match dataset
			// CNPJ
			[false, '39484341348596'],
			[false, '02075164172747'],
			[false, '79537905668250'],
			[false, '63267877400743'],
			[false, '80364193180839'],
			[false, '82241205738198'],
			[false, '12637960301913'],
			[false, '20469362896000'],
			[false, '08024616923279'],
			[false, '69039361143238'],
			[false, '71720374965335'],
			[false, '52154842487821'],
			[false, '02758742406607'],
			[false, '58865634972707'],
			[false, '63211841031338'],
			[false, '94138562095150'],
			[false, '06566050995591'],
			[false, '01297678590130'],
			[false, '94075457641002'],
			[false, '45124178042817'],
			[false, '43957971973121'],
			[false, '72891434285556'],
			[false, '10367479348542'],
			[false, '64308935791313'],
			[false, '24690537393640'],
			[false, '35371639481704'],
			[false, '70432695957323'],
			[false, '71444903496255'],
			[false, '16360225241903'],
			[false, '78054306152937'],
			[false, '22178671151035'],
			[false, '45217683788660'],
			[false, '43223714862301'],
			[false, '01094921498314'],
			[false, '41735397140928'],
			[false, '61427335482182'],
			[false, '13872290960471'],
			[false, '97112742683513'],
			[false, '33782071462348'],
			[false, '93014855491651'],
			[false, '60152926244057'],
			[false, '28995421776255'],
			[false, '09065066257639'],
			[false, '04096246113858'],
			[false, '04977807746089'],
			[false, '73990312254638'],
			[false, '65071892492324'],
			[false, '51812368579963'],
			[false, '27664045740338'],
			[false, '80288570222052'],
			// CPF
			[false, '72586202777'],
			[false, '81612473823'],
			[false, '44271953753'],
			[false, '87310966785'],
			[false, '97081948529'],
			[false, '43516157478'],
			[false, '11814014688'],
			[false, '55139518124'],
			[false, '20429640917'],
			[false, '80309888147'],
			[false, '82417037846'],
			[false, '12912907260'],
			[false, '39389557243'],
			[false, '91631208531'],
			[false, '85147712512'],
			[false, '13815465037'],
			[false, '71938650513'],
			[false, '90982877911'],
			[false, '28785794749'],
			[false, '63506168799'],
			[false, '49815932998'],
			[false, '08976770172'],
			[false, '92349453108'],
			[false, '19045203568'],
			[false, '16841446270'],
			[false, '38524862173'],
			[false, '40250584490'],
			[false, '23798243886'],
			[false, '99357979250'],
			[false, '39203444820'],
			[false, '09086339285'],
			[false, '47198972953'],
			[false, '64584130853'],
			[false, '73017026363'],
			[false, '06727369304'],
			[false, '54917969046'],
			[false, '45379813343'],
			[false, '95604195675'],
			[false, '44298415573'],
			[false, '60260679165'],
			[false, '91719821762'],
			[false, '43074689432'],
			[false, '70623243018'],
			[false, '47265624072'],
			[false, '32601935285'],
			[false, '95139700617'],
			[false, '35067516686'],
			[false, '31027505880'],
			[false, '64745125374'],
			[false, '97291338690'],
		];
	}

	/**
	 * Assert if $email is validated as $expected.
	 *
	 * @covers ::parseEmail
	 * @covers ::validateEmail
	 * @dataProvider dataEmail
	 * @test Expecting positive assertion.
	 * @param boolean $expected Expected result.
	 * @param boolean $email Email to validate
	 * @return boolean
	 */
	public function canValidateEmail ( bool $expected, string $email )
	{ $this->assertEquals($expected, Parser::validateEmail($email)); }

	/**
	 * A list with random emails to validate.
	 * Provider to canValidateEmail() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataEmail () : array
	{
		return [
			// Must match dataset
			// E-MAILS
			[true, 'lais.dias@faro.com'],
			[true, 'melissa09@ig.com.br'],
			[true, 'noel17@yahoo.com'],
			[true, 'isabel97@roque.com.br'],
			[true, 'marcio82@hotmail.com'],
			[true, 'barreto.luan@rocha.com.br'],
			[true, 'elisa73@benites.com'],
			[true, 'sara.colaco@feliciano.br'],
			[true, 'daniella63@roque.com.br'],
			[true, 'ricardo.matias@ig.com.br'],
			[true, 'balestero.ketlin@queiros.org'],
			[true, 'casanova.hugo@hotmail.com'],
			[true, 'bella87@uol.com.br'],
			[true, 'hsaraiva@roque.com'],
			[true, 'santana.maximiano@uol.com.br'],
			[true, 'fgomes@aranda.com'],
			[true, 'batista.rodrigo@medina.com'],
			[true, 'jcamacho@queiros.net.br'],
			[true, 'ortiz.jacomo@r7.com'],
			[true, 'santiago24@gomes.com'],
			[true, 'qpaes@hotmail.com'],
			[true, 'sverdara@hotmail.com'],
			[true, 'brito.luciana@gmail.com'],
			[true, 'urias.bruno@molina.com.br'],
			[true, 'galhardo.jonas@r7.com'],
			[true, 'miranda17@flores.org'],
			[true, 'rvalentin@ig.com.br'],
			[true, 'julieta.serra@ortega.com'],
			[true, 'prado.yasmin@matos.com.br'],
			[true, 'isaac.santiago@ig.com.br'],
			[true, 'sgalhardo@dasilva.net.br'],
			[true, 'allan.ortega@correia.net'],
			[true, 'mateus27@dasilva.org'],
			[true, 'vega.julia@r7.com'],
			[true, 'jorge61@terra.com.br'],
			[true, 'elisa.mares@yahoo.com'],
			[true, 'fernandes.aaron@uol.com.br'],
			[true, 'benedito10@yahoo.com'],
			[true, 'julia.salas@hotmail.com'],
			[true, 'ortiz.danilo@hotmail.com'],
			[true, 'jrezende@r7.com'],
			[true, 'benjamin51@defreitas.net'],
			[true, 'luciano66@r7.com'],
			[true, 'hugo.garcia@hotmail.com'],
			[true, 'hdarosa@hotmail.com'],
			[true, 'azevedo.aaron@terra.com.br'],
			[true, 'yasmin84@dasneves.net'],
			[true, 'vpadilha@molina.br'],
			[true, 'jean.valencia@gmail.com'],
			[true, 'estela59@ig.com.br'],
			[true, 'valencia.richard@example.com'],
			[true, 'salazar.martinho@example.org'],
			[true, 'souza.thalita@example.com'],
			[true, 'lpena@example.org'],
			[true, 'andres46@example.net'],
			[true, 'leticia94@example.net'],
			[true, 'adasneves@example.net'],
			[true, 'queiros.agustina@example.com'],
			[true, 'catarina.batista@example.com'],
			[true, 'juliana.rico@example.com'],
			[true, 'zambrano.miguel@example.net'],
			[true, 'maia.noeli@example.org'],
			[true, 'emilia97@example.net'],
			[true, 'tatiane.balestero@example.org'],
			[true, 'igil@example.com'],
			[true, 'mel63@example.com'],
			[true, 'kdeaguiar@example.com'],
			[true, 'gomes.sandro@example.net'],
			[true, 'mfernandes@example.org'],
			[true, 'xqueiros@example.net'],
			[true, 'rivera.theo@example.net'],
			[true, 'lilian.toledo@example.org'],
			[true, 'cesar.fontes@example.org'],
			[true, 'artur.sanches@example.net'],
			[true, 'pablo.deoliveira@example.net'],
			[true, 'mferminiano@example.com'],
			[true, 'madeira.mario@example.net'],
			[true, 'vdeverso@example.net'],
			[true, 'qferreira@example.org'],
			[true, 'carrara.afonso@example.com'],
			[true, 'ellen58@example.org'],
			[true, 'jacomo.carvalho@example.org'],
			[true, 'bqueiros@example.org'],
			[true, 'btamoio@example.net'],
			[true, 'hescobar@example.com'],
			[true, 'yohanna.deaguiar@example.com'],
			[true, 'xverdugo@example.net'],
			[true, 'daiane.deverso@example.net'],
			[true, 'zaragoca.rogerio@example.net'],
			[true, 'thiago60@example.org'],
			[true, 'ocolaco@example.com'],
			[true, 'ester06@example.org'],
			[true, 'tamoio.martinho@example.com'],
			[true, 'carmona.cauan@example.net'],
			[true, 'katherine79@example.com'],
			[true, 'nquintana@example.com'],
			[true, 'victor58@example.net'],
			[true, 'diego18@example.com'],
			[true, 'jperez@example.com'],
			[true, 'eloa80@example.org'],
			[true, 'prado.nelson@r7.com'],
			[true, 'gisela.lovato@r7.com'],
			[true, 'valente.emanuel@gmail.com'],
			[true, 'bgalindo@uol.com.br'],
			[true, 'ian.teles@terra.com.br'],
			[true, 'thomas.oliveira@r7.com'],
			[true, 'beltrao.juliana@hotmail.com'],
			[true, 'feliciano.reinaldo@terra.com.br'],
			[true, 'ian72@ig.com.br'],
			[true, 'molina.lia@gmail.com'],
			[true, 'valentin.clarice@ig.com.br'],
			[true, 'joaquim73@gmail.com'],
			[true, 'pamela23@uol.com.br'],
			[true, 'dasilva.daniel@gmail.com'],
			[true, 'deivid43@yahoo.com'],
			[true, 'patricia.maldonado@yahoo.com'],
			[true, 'gabrielle60@r7.com'],
			[true, 'queiros.kauan@ig.com.br'],
			[true, 'elis.martines@yahoo.com'],
			[true, 'gpereira@gmail.com'],
			[true, 'msouza@r7.com'],
			[true, 'olga14@yahoo.com'],
			[true, 'feliciano.sophie@gmail.com'],
			[true, 'adriel.cruz@hotmail.com'],
			[true, 'emerson.serra@r7.com'],
			[true, 'luiz74@r7.com'],
			[true, 'nicolas70@hotmail.com'],
			[true, 'ramos.rogerio@ig.com.br'],
			[true, 'xsalas@yahoo.com'],
			[true, 'moises10@uol.com.br'],
			[true, 'zambrano.alma@terra.com.br'],
			[true, 'dominato.dener@gmail.com'],
			[true, 'feliciano.inacio@r7.com'],
			[true, 'reinaldo57@uol.com.br'],
			[true, 'santacruz.sandra@yahoo.com'],
			[true, 'sophie.batista@terra.com.br'],
			[true, 'vitoria83@ig.com.br'],
			[true, 'juliano.valencia@uol.com.br'],
			[true, 'lvaldez@yahoo.com'],
			[true, 'franco.flor@gmail.com'],
			[true, 'kmaia@ig.com.br'],
			[true, 'joaquin80@ig.com.br'],
			[true, 'czaragoca@uol.com.br'],
			[true, 'qsoares@ig.com.br'],
			[true, 'dominato.antonella@r7.com'],
			[true, 'eloa.santana@uol.com.br'],
			[true, 'horacio32@terra.com.br'],
			[true, 'alessandra36@gmail.com'],
			[true, 'cynthia.amaral@gmail.com'],
			[true, 'thalissa.escobar@ig.com.br'],
			[true, 'aurora.rosa assuncao.br'],
			[true, 'aranda.anderson mares.net.br'],
			[true, 'qaragao r7.com'],
			[true, 'qsouza marques.com'],
			[true, 'wilson.carvalho gil.com.br'],
			[true, 'martinho97 bonilha.com'],
			[true, 'vdelatorre rivera.com'],
			[true, 'alana.goncalves gmail.com'],
			[true, 'thales.uchoa mendes.net.br'],
			[true, 'ian53 terra.com.br'],
			[true, 'galindo.willian vega.com.br'],
			[true, 'thiago48 medina.com.br'],
			[true, 'camacho.felipe r7.com'],
			[true, 'qalves espinoza.net.br'],
			[true, 'rserra perez.com'],
			[true, 'oestrada ig.com.br'],
			[true, 'sales.everton esteves.com'],
			[true, 'cordeiro.cristiana terra.com.br'],
			[true, 'reis.teo zamana.org'],
			[true, 'goncalves.mauro r7.com'],
			[true, 'cervantes.pietra mendes.org'],
			[true, 'caldeira.marta dasdores.org'],
			[true, 'jfranco gmail.com'],
			[true, 'yserrano uol.com.br'],
			[true, 'teles.jeronimo martines.org'],
			[true, 'ycaldeira r7.com'],
			[true, 'vmarques aragao.br'],
			[true, 'thalia.ortega yahoo.com'],
			[true, 'tdomingues yahoo.com'],
			[true, 'saito.simon cortes.net.br'],
			[true, 'aaron46 ig.com.br'],
			[true, 'fonseca.mirella santana.com.br'],
			[true, 'gomes.mauro ig.com.br'],
			[true, 'balves yahoo.com'],
			[true, 'maiara.gomes gmail.com'],
			[true, 'suellen83 sandoval.br'],
			[true, 'cristiano.feliciano sanches.net'],
			[true, 'galindo.mario martines.com.br'],
			[true, 'zguerra marques.org'],
			[true, 'gcordeiro delgado.net.br'],
			[true, 'zferreira romero.net.br'],
			[true, 'alana85 terra.com.br'],
			[true, 'vale.walter r7.com'],
			[true, 'dias.emanuel ig.com.br'],
			[true, 'suzana.fontes r7.com'],
			[true, 'laiane99 paz.net'],
			[true, 'bmeireles hotmail.com'],
			[true, 'gustavo31 uol.com.br'],
			[true, 'asalazar madeira.br'],
			[true, 'ohana78 ramos.com.br'],
			[true, 'verdugo.noel example.org'],
			[true, 'isaac.padrao example.net'],
			[true, 'natan77 example.org'],
			[true, 'fatima69 example.net'],
			[true, 'gilberto36 example.com'],
			[true, 'vneves example.com'],
			[true, 'antonella.duarte example.com'],
			[true, 'soto.jose example.net'],
			[true, 'bcampos example.org'],
			[true, 'vale.alexa example.net'],
			[true, 'duarte.inacio example.com'],
			[true, 'paulo80 example.org'],
			[true, 'murilo.barreto example.org'],
			[true, 'kgil example.com'],
			[true, 'karanda example.org'],
			[true, 'miguel16 example.org'],
			[true, 'alexandre.medina example.org'],
			[true, 'meireles.miriam example.org'],
			[true, 'benedito58 example.org'],
			[true, 'afurtado example.org'],
			[true, 'leo43 example.com'],
			[true, 'nathalia72 example.org'],
			[true, 'grego.tatiana example.net'],
			[true, 'everton.tamoio example.org'],
			[true, 'burgos.camila example.net'],
			[true, 'kuchoa example.net'],
			[true, 'dserrano example.org'],
			[true, 'afeliciano example.net'],
			[true, 'zsaraiva example.com'],
			[true, 'benicio94 example.com'],
			[true, 'valentin.santana example.com'],
			[true, 'lvega example.com'],
			[true, 'sueli41 example.net'],
			[true, 'hortencia.rios example.com'],
			[true, 'rbeltrao example.org'],
			[true, 'emerson.soares example.org'],
			[true, 'edson.saito example.net'],
			[true, 'jrivera example.net'],
			[true, 'vdefreitas example.com'],
			[true, 'karen68 example.net'],
			[true, 'montenegro.gabrielle example.com'],
			[true, 'aurora90 example.org'],
			[true, 'jefferson69 example.org'],
			[true, 'nayara.casanova example.org'],
			[true, 'madeira.lais example.com'],
			[true, 'sandro98 example.net'],
			[true, 'everton48 example.net'],
			[true, 'jferreira example.com'],
			[true, 'ferreira.luana example.net'],
			[true, 'reinaldo.fernandes example.org'],
			[true, 'aguiar.arthur uol.com.br'],
			[true, 'valdez.leticia ig.com.br'],
			[true, 'pena.joana ig.com.br'],
			[true, 'sergio64 yahoo.com'],
			[true, 'aranda.emily uol.com.br'],
			[true, 'ksalas r7.com'],
			[true, 'delgado.antonio terra.com.br'],
			[true, 'kevin.marques gmail.com'],
			[true, 'wgalhardo gmail.com'],
			[true, 'denise.serra uol.com.br'],
			[true, 'renata18 gmail.com'],
			[true, 'martinho.oliveira r7.com'],
			[true, 'mmolina terra.com.br'],
			[true, 'thales30 gmail.com'],
			[true, 'giovanna.carvalho r7.com'],
			[true, 'gael17 yahoo.com'],
			[true, 'fidalgo.wagner hotmail.com'],
			[true, 'goncalves.cynthia hotmail.com'],
			[true, 'flavio.sales uol.com.br'],
			[true, 'katherine17 ig.com.br'],
			[true, 'jose.saito hotmail.com'],
			[true, 'lorena11 gmail.com'],
			[true, 'tamoio.jorge ig.com.br'],
			[true, 'ymartines yahoo.com'],
			[true, 'dayane95 r7.com'],
			[true, 'danielle95 hotmail.com'],
			[true, 'fabiana.rangel terra.com.br'],
			[true, 'bonilha.violeta yahoo.com'],
			[true, 'cristiano18 r7.com'],
			[true, 'nelson77 uol.com.br'],
			[true, 'claudia70 terra.com.br'],
			[true, 'benedito.maldonado terra.com.br'],
			[true, 'alcantara.aaron uol.com.br'],
			[true, 'franciele86 terra.com.br'],
			[true, 'fsandoval r7.com'],
			[true, 'dbalestero gmail.com'],
			[true, 'ncamacho terra.com.br'],
			[true, 'bittencourt.cezar gmail.com'],
			[true, 'aguiar.henrique terra.com.br'],
			[true, 'kamila.leon terra.com.br'],
			[true, 'mariana.burgos yahoo.com'],
			[true, 'iserra ig.com.br'],
			[true, 'cortiz uol.com.br'],
			[true, 'artur.barros terra.com.br'],
			[true, 'amelia00 uol.com.br'],
			[true, 'ucaldeira ig.com.br'],
			[true, 'joyce.serra ig.com.br'],
			[true, 'rescobar ig.com.br'],
			[true, 'jean95 yahoo.com'],
			[true, 'rosa.sebastiao gmail.com'],
			// Must not match dataset
			// E-MAILS
			[false, 'antoniotamoiosolanocom'],
			[false, 'ferreiralorenacamposcom'],
			[false, 'miladeaguiar@yahoocom'],
			[false, 'luzia76gmail.com'],
			[false, 'wellingtonsoto@hotmailcom'],
			[false, 'lealgiovana@quintanabr'],
			[false, 'matiasarturpazbr'],
			[false, 'tbezerra@igcombr'],
			[false, 'ohana62@penanetbr'],
			[false, 'denislourencohotmailcom'],
			[false, 'nicole62@gmailcom'],
			[false, 'wilson28@r7com'],
			[false, 'yserranocamachocom'],
			[false, 'dasneveseduardogmailcom'],
			[false, 'fonsecaandrebenezcombr'],
			[false, 'espinoza.valentinar7.com'],
			[false, 'bruno77hotmailcom'],
			[false, 'leonardocolacobonilhanet'],
			[false, 'nicolas91avilacom'],
			[false, 'antonio05@espinozaorg'],
			[false, 'lorenzo22gmail.com'],
			[false, 'fpontesig.com.br'],
			[false, 'cricoterra.com.br'],
			[false, 'darosaagathaterracombr'],
			[false, 'valestellar7com'],
			[false, 'tamaralsernabr'],
			[false, 'bruno.bezerralutero.com'],
			[false, 'fidalgomilene@yahoocom'],
			[false, 'xferrazfernandescom'],
			[false, 'gbrito@terracombr'],
			[false, 'louise59@r7com'],
			[false, 'ulirar7.com'],
			[false, 'pazeduardor7com'],
			[false, 'uavilar7.com'],
			[false, 'adeoliveirar7com'],
			[false, 'sserraig.com.br'],
			[false, 'xgalindo@alvescombr'],
			[false, 'pgodoi@terracombr'],
			[false, 'toledo.simaobarros.net'],
			[false, 'gilziraldo@burgoscom'],
			[false, 'giovana02desouza.org'],
			[false, 'deversoantonella@hotmailcom'],
			[false, 'waranda@hotmailcom'],
			[false, 'wbarreto@yahoocom'],
			[false, 'ortegaconstancia@yahoocom'],
			[false, 'eroque@marquescom'],
			[false, 'pedrocamposrosacombr'],
			[false, 'tcamachogmail.com'],
			[false, 'dbonilhagmail.com'],
			[false, 'pespinozafaro.br'],
			[false, 'delvalle.hernanipadilha.com.br'],
			[false, 'valente.laianeyahoo.com'],
			[false, 'vitoria69zaragoca.org'],
			[false, 'juliana66vieira.net.br'],
			[false, 'lsanchesbeltraonet'],
			[false, 'lpadilha@espinozacombr'],
			[false, 'jose25godoi.br'],
			[false, 'tpereiraig.com.br'],
			[false, 'rosageanterracombr'],
			[false, 'claricesalazarr7com'],
			[false, 'bperezigcombr'],
			[false, 'pontesalessandra@ferrazcombr'],
			[false, 'apaesyahoo.com'],
			[false, 'nelsondesouzauolcombr'],
			[false, 'marcosvelasquesigcombr'],
			[false, 'horacioriverazambranoorg'],
			[false, 'aviladanielsalesorg'],
			[false, 'umarinmadeirabr'],
			[false, 'flaviaarruda@terracombr'],
			[false, 'williamvalenciasandovalcombr'],
			[false, 'tais.vieirar7.com'],
			[false, 'sandro57terracombr'],
			[false, 'bernardogalindo@terracombr'],
			[false, 'lia89santacruz.net'],
			[false, 'zambranomaraisavalentinnetbr'],
			[false, 'luis63burgoscombr'],
			[false, 'mateus76terracombr'],
			[false, 'santanaantonieta@cortesnet'],
			[false, 'gtoledo@serracombr'],
			[false, 'cauanpedrosaterracombr'],
			[false, 'karina94@dasnevesnet'],
			[false, 'daiane44@hotmailcom'],
			[false, 'martinhoroque@ortegaorg'],
			[false, 'ysalazargarcia.br'],
			[false, 'salasrogerioyahoocom'],
			[false, 'jacomosandovalescobarorg'],
			[false, 'olivia93gmail.com'],
			[false, 'sernaruth@uolcombr'],
			[false, 'noel34@terracombr'],
			[false, 'gusmao.joaquimyahoo.com'],
			[false, 'abgail.romerouol.com.br'],
			[false, 'louise05hotmailcom'],
			[false, 'jefferson82terra.com.br'],
			[false, 'solanomarilia@terracombr'],
			[false, 'ivana67@uolcombr'],
			[false, 'lucio17@terracombr'],
			[false, 'teo47@uolcombr'],
			[false, 'dias.kesiauol.com.br'],
			[false, 'ecarvalho@yahoocom'],
			[false, 'cauan.fontespedrosa.org'],
			[false, 'itamoiodarosaorg'],
			[false, 'barbara.jimeneshotmail.com'],
			[false, 'laiane33coronanet'],
			[false, 'quintanarafaeldasnevesnet'],
			[false, 'faro.patriciayahoo.com'],
			[false, 'leandrofontes@terracombr'],
			[false, 'barros.luismendes.net.br'],
			[false, 'teosalasgmailcom'],
			[false, 'burgosmichaelvaldezorg'],
			[false, 'ferminianogeorge@terracombr'],
			[false, 'eloa51cordeirocombr'],
			[false, 'vitor44@yahoocom'],
			[false, 'dpenamontenegronetbr'],
			[false, 'valenicoleyahoocom'],
			[false, 'leonestela@gmailcom'],
			[false, 'darosa.karineuol.com.br'],
			[false, 'nguerra@uolcombr'],
			[false, 'floreslucas@estradacom'],
			[false, 'carraraigorvieiracom'],
			[false, 'sandovalsantiago@diascom'],
			[false, 'quintanabetina@hotmailcom'],
			[false, 'lia26marques.br'],
			[false, 'rodolfo.mendeshotmail.com'],
			[false, 'roque.yohannarodrigues.com'],
			[false, 'rodrigues.ivanterra.com.br'],
			[false, 'ssolano@yahoocom'],
			[false, 'daiane50@yahoocom'],
			[false, 'santacruz.victorterra.com.br'],
			[false, 'avilamauro@r7com'],
			[false, 'glutero@igcombr'],
			[false, 'pbeltrao@igcombr'],
			[false, 'david80@cortescombr'],
			[false, 'kfidalgogmailcom'],
			[false, 'cristovao31matias.net'],
			[false, 'madalenabarrosbalesteronetbr'],
			[false, 'gabriel60terra.com.br'],
			[false, 'jassuncaohotmailcom'],
			[false, 'bittencourtfabriciodasnevesorg'],
			[false, 'sarah60terra.com.br'],
			[false, 'teobaldoferreira@montenegrocom'],
			[false, 'maximo.dasilvaserrano.com.br'],
			[false, 'vega.kauanassuncao.com.br'],
			[false, 'slovatoespinozabr'],
			[false, 'queiros.micheleuol.com.br'],
			[false, 'daiane.paesuol.com.br'],
			[false, 'ricardo30terra.com.br'],
			[false, 'cristianogarcia@uolcombr'],
			[false, 'flores.giovanerico.net.br'],
			[false, 'mescobar@yahoocom'],
			[false, 'coronarosana@yahoocom'],
		];
	}

	/**
	 * Assert if $phone is validated as $expected.
	 *
	 * @covers ::parsePhone
	 * @covers ::validatePhone
	 * @dataProvider dataPhone
	 * @test Expecting positive assertion.
	 * @param boolean $expected Expected result.
	 * @param boolean $phone Phone to validate
	 * @return boolean
	 */
	public function canValidatePhone ( bool $expected, string $phone )
	{ $this->assertEquals($expected, Parser::validatePhone($phone)); }

	/**
	 * A list with random phones to validate.
	 * Provider to canValidatePhone() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataPhone () : array
	{
		return [
			// Must match dataset
			// PHONES
			[true, '(32) 98677-4375'],
			[true, '(45) 2457-5199'],
			[true, '(69) 93361-1559'],
			[true, '(63) 3618-8995'],
			[true, '(86) 97224-2725'],
			[true, '(89) 99230-1156'],
			[true, '(88) 3246-2200'],
			[true, '(91) 3866-3281'],
			[true, '(33) 91259-9680'],
			[true, '(61) 95697-6765'],
			[true, '(91) 4282-9325'],
			[true, '(92) 98775-6852'],
			[true, '(31) 94501-0684'],
			[true, '(61) 2592-0800'],
			[true, '(41) 3765-6949'],
			[true, '(13) 99827-4028'],
			[true, '(18) 96878-1872'],
			[true, '(67) 93087-9486'],
			[true, '(68) 4323-4623'],
			[true, '(13) 97738-5616'],
			[true, '(53) 2509-3222'],
			[true, '(31) 92126-1505'],
			[true, '(44) 98044-7702'],
			[true, '(53) 3893-7008'],
			[true, '(18) 4441-3293'],
			[true, '(38) 99857-6440'],
			[true, '(32) 2800-6585'],
			[true, '(33) 4589-6104'],
			[true, '(11) 94537-1792'],
			[true, '(33) 92701-4402'],
			[true, '(37) 92934-5180'],
			[true, '(85) 93684-0670'],
			[true, '(12) 4894-0395'],
			[true, '(74) 99199-6366'],
			[true, '(34) 2509-3231'],
			[true, '(97) 4878-0151'],
			[true, '(47) 2015-5015'],
			[true, '(44) 99352-0703'],
			[true, '(21) 2304-8435'],
			[true, '(27) 97677-8114'],
			[true, '(92) 96224-6662'],
			[true, '(14) 3885-0748'],
			[true, '(12) 90586-7421'],
			[true, '(71) 4872-0970'],
			[true, '(88) 95532-1452'],
			[true, '(55) 4766-9546'],
			[true, '(42) 3779-7311'],
			[true, '(35) 95843-8731'],
			[true, '(17) 4967-7136'],
			[true, '(95) 99151-1696'],
			[true, '43962435565'],
			[true, '97975522174'],
			[true, '8738410787'],
			[true, '8129198119'],
			[true, '84920258180'],
			[true, '46986651737'],
			[true, '6337816230'],
			[true, '99903927823'],
			[true, '92935996450'],
			[true, '53983031889'],
			[true, '21953944914'],
			[true, '9145615537'],
			[true, '22965155693'],
			[true, '74994633426'],
			[true, '1941321184'],
			[true, '97938625755'],
			[true, '55984212399'],
			[true, '43932587090'],
			[true, '1945584020'],
			[true, '86927115462'],
			[true, '6740187984'],
			[true, '63930945684'],
			[true, '96972451298'],
			[true, '6424397168'],
			[true, '5343982429'],
			[true, '6929893843'],
			[true, '1232209021'],
			[true, '64996134309'],
			[true, '46943395714'],
			[true, '1731711323'],
			[true, '33922724264'],
			[true, '6731347955'],
			[true, '95989328231'],
			[true, '9641990162'],
			[true, '4632721785'],
			[true, '7725413389'],
			[true, '7737202940'],
			[true, '63956726937'],
			[true, '89929446334'],
			[true, '9536051823'],
			[true, '4247249816'],
			[true, '16915073639'],
			[true, '48934861643'],
			[true, '4327839994'],
			[true, '7134526278'],
			[true, '1322347187'],
			[true, '4725094076'],
			[true, '45980137683'],
			[true, '1226022383'],
			[true, '67997354809'],
			[true, '+55 (63) 3384-8698'],
			[true, '+55 (95) 90226-4074'],
			[true, '+55 (55) 3537-5177'],
			[true, '+55 (75) 99316-1010'],
			[true, '+55 (64) 2022-8762'],
			[true, '+55 (96) 98252-4428'],
			[true, '+55 (24) 2804-5237'],
			[true, '+55 (43) 91446-6599'],
			[true, '+55 (91) 4338-6697'],
			[true, '+55 (24) 4351-1337'],
			[true, '+55 (71) 91744-9914'],
			[true, '+55 (43) 93015-8206'],
			[true, '+55 (86) 95024-6147'],
			[true, '+55 (71) 4371-8553'],
			[true, '+55 (28) 99357-3536'],
			[true, '+55 (62) 92330-4682'],
			[true, '+55 (85) 92211-4236'],
			[true, '+55 (87) 3060-0674'],
			[true, '+55 (51) 93864-6712'],
			[true, '+55 (71) 91155-5221'],
			[true, '+55 (38) 2023-0716'],
			[true, '+55 (96) 97475-1531'],
			[true, '+55 (79) 97967-3183'],
			[true, '+55 (33) 98156-1312'],
			[true, '+55 (46) 4173-3164'],
			[true, '+55 (98) 4227-9836'],
			[true, '+55 (55) 90494-8219'],
			[true, '+55 (98) 4247-8230'],
			[true, '+55 (17) 96558-7553'],
			[true, '+55 (97) 95421-5211'],
			[true, '+55 (27) 3286-2438'],
			[true, '+55 (12) 3150-8927'],
			[true, '+55 (53) 93475-0282'],
			[true, '+55 (79) 4649-7231'],
			[true, '+55 (67) 4556-3831'],
			[true, '+55 (97) 3666-9724'],
			[true, '+55 (24) 2581-2096'],
			[true, '+55 (94) 4629-5837'],
			[true, '+55 (54) 2575-1483'],
			[true, '+55 (53) 4188-4878'],
			[true, '+55 (75) 2559-0600'],
			[true, '+55 (82) 3829-4105'],
			[true, '+55 (99) 97790-0294'],
			[true, '+55 (85) 2793-0725'],
			[true, '+55 (14) 92705-9631'],
			[true, '+55 (14) 90037-3930'],
			[true, '+55 (16) 2203-4676'],
			[true, '+55 (46) 98794-2280'],
			[true, '+55 (16) 2434-7429'],
			[true, '+55 (81) 94293-2444'],
			[true, '+55 14983164230'],
			[true, '+55 49981112125'],
			[true, '+55 5543406583'],
			[true, '+55 31942025826'],
			[true, '+55 74968199248'],
			[true, '+55 1827293097'],
			[true, '+55 81935584319'],
			[true, '+55 49904355827'],
			[true, '+55 27941821038'],
			[true, '+55 84960931880'],
			[true, '+55 91952605232'],
			[true, '+55 38936106041'],
			[true, '+55 1438827040'],
			[true, '+55 33914760547'],
			[true, '+55 8928670201'],
			[true, '+55 9641883655'],
			[true, '+55 3334898077'],
			[true, '+55 3723247580'],
			[true, '+55 31902426256'],
			[true, '+55 16975436318'],
			[true, '+55 55979640072'],
			[true, '+55 55995319765'],
			[true, '+55 91954910191'],
			[true, '+55 31996531845'],
			[true, '+55 64906741850'],
			[true, '+55 55924701957'],
			[true, '+55 65950732987'],
			[true, '+55 21949508151'],
			[true, '+55 63943939666'],
			[true, '+55 8844087173'],
			[true, '+55 6829145019'],
			[true, '+55 8439387758'],
			[true, '+55 73928867534'],
			[true, '+55 49977687622'],
			[true, '+55 15901998427'],
			[true, '+55 4120486709'],
			[true, '+55 8323781997'],
			[true, '+55 4933828819'],
			[true, '+55 9847208589'],
			[true, '+55 1845963404'],
			[true, '+55 97905769974'],
			[true, '+55 88935239449'],
			[true, '+55 9349197811'],
			[true, '+55 63942597718'],
			[true, '+55 8239460540'],
			[true, '+55 6437973888'],
			[true, '+55 15950412597'],
			[true, '+55 5144112680'],
			[true, '+55 44974748636'],
			[true, '+55 31926186586'],
			// Must not match dataset
			// PHONES WITHOUT AREACODE
			[false, '95922-7919'],
			[false, '93594-6347'],
			[false, '96177-8838'],
			[false, '93014-1316'],
			[false, '95257-6410'],
			[false, '99047-5211'],
			[false, '96929-5663'],
			[false, '98632-0073'],
			[false, '98352-0173'],
			[false, '94747-7459'],
			[false, '98473-4627'],
			[false, '90846-3614'],
			[false, '94331-3467'],
			[false, '99802-2473'],
			[false, '98211-8234'],
			[false, '96621-0474'],
			[false, '93088-2529'],
			[false, '92998-4993'],
			[false, '96809-7586'],
			[false, '92981-2786'],
			[false, '98323-6139'],
			[false, '94809-5483'],
			[false, '91047-2602'],
			[false, '95267-2144'],
			[false, '97537-1356'],
			[false, '99509-2927'],
			[false, '91409-8480'],
			[false, '96197-4579'],
			[false, '93713-4984'],
			[false, '99391-8764'],
			[false, '94539-0591'],
			[false, '96947-9172'],
			[false, '96626-9977'],
			[false, '90786-9081'],
			[false, '94760-3890'],
			[false, '90166-1004'],
			[false, '90401-5918'],
			[false, '94793-5528'],
			[false, '93301-5377'],
			[false, '98021-5344'],
			[false, '94033-1617'],
			[false, '98911-3483'],
			[false, '95025-3377'],
			[false, '94430-2169'],
			[false, '93624-7151'],
			[false, '95672-3205'],
			[false, '92894-0590'],
			[false, '95167-0449'],
			[false, '94719-9915'],
			[false, '93749-3754'],
			[false, '2430-3831'],
			[false, '3779-3078'],
			[false, '4615-7667'],
			[false, '4642-2545'],
			[false, '4072-7145'],
			[false, '2540-9097'],
			[false, '2013-8332'],
			[false, '2514-7804'],
			[false, '3245-1653'],
			[false, '4753-5281'],
			[false, '3902-7855'],
			[false, '3507-9305'],
			[false, '3011-6743'],
			[false, '4281-3173'],
			[false, '4672-9336'],
			[false, '2887-9125'],
			[false, '2926-3203'],
			[false, '3771-0780'],
			[false, '2234-3144'],
			[false, '2190-6187'],
			[false, '4541-4770'],
			[false, '3151-9096'],
			[false, '4761-0389'],
			[false, '3906-3164'],
			[false, '2838-6478'],
			[false, '3700-9947'],
			[false, '3770-1321'],
			[false, '3726-8165'],
			[false, '4079-2200'],
			[false, '3050-0121'],
			[false, '3917-8804'],
			[false, '3169-7146'],
			[false, '3924-9006'],
			[false, '4763-0340'],
			[false, '2928-9449'],
			[false, '3949-0721'],
			[false, '4719-0317'],
			[false, '3424-4356'],
			[false, '3441-9202'],
			[false, '3652-9062'],
			[false, '4448-8929'],
			[false, '2701-1961'],
			[false, '3143-9292'],
			[false, '2292-6319'],
			[false, '4059-5113'],
			[false, '3148-1955'],
			[false, '4309-2827'],
			[false, '3548-7439'],
			[false, '3130-0117'],
			[false, '4385-2657'],
			[false, '+5598909-2712'],
			[false, '+5598374-2734'],
			[false, '+5599151-5474'],
			[false, '+5598206-6978'],
			[false, '+5599105-0009'],
			[false, '+5590978-6378'],
			[false, '+5597564-8378'],
			[false, '+5594919-1238'],
			[false, '+5591910-0703'],
			[false, '+5594256-1613'],
			[false, '+5597753-2816'],
			[false, '+5598854-6508'],
			[false, '+5591191-1650'],
			[false, '+5599322-4759'],
			[false, '+5597996-3620'],
			[false, '+5599819-4699'],
			[false, '+5590525-9540'],
			[false, '+5593246-0144'],
			[false, '+5591011-2994'],
			[false, '+5596972-0862'],
			[false, '+5598617-6800'],
			[false, '+5598846-9412'],
			[false, '+5594301-0154'],
			[false, '+5590283-5898'],
			[false, '+5599276-4630'],
			[false, '+5598591-3249'],
			[false, '+5598983-6813'],
			[false, '+5598922-0827'],
			[false, '+5599151-7620'],
			[false, '+5598899-9035'],
			[false, '+5598550-2394'],
			[false, '+5593255-2844'],
			[false, '+5595741-1861'],
			[false, '+5594462-2975'],
			[false, '+5599024-1209'],
			[false, '+5596864-0284'],
			[false, '+5597011-6433'],
			[false, '+5593438-5620'],
			[false, '+5590084-4765'],
			[false, '+5591815-3416'],
			[false, '+5594712-3225'],
			[false, '+5597075-4503'],
			[false, '+5593382-9828'],
			[false, '+5594245-0100'],
			[false, '+5590375-8834'],
			[false, '+5597884-6531'],
			[false, '+5593290-3869'],
			[false, '+5594158-3677'],
			[false, '+5595797-7586'],
			[false, '+5592824-6840'],
			[false, '+552704-9936'],
			[false, '+554279-6653'],
			[false, '+552508-3783'],
			[false, '+552902-3538'],
			[false, '+552347-1659'],
			[false, '+553347-8819'],
			[false, '+553233-6471'],
			[false, '+554381-4802'],
			[false, '+554317-8754'],
			[false, '+552745-3979'],
			[false, '+552932-1523'],
			[false, '+554571-9655'],
			[false, '+553835-6639'],
			[false, '+554305-7455'],
			[false, '+554632-5238'],
			[false, '+552296-6592'],
			[false, '+552868-6660'],
			[false, '+553173-1207'],
			[false, '+552433-0594'],
			[false, '+552641-3338'],
			[false, '+553524-7364'],
			[false, '+553496-5576'],
			[false, '+552362-1029'],
			[false, '+554325-0832'],
			[false, '+553224-5947'],
			[false, '+553125-0921'],
			[false, '+554288-0432'],
			[false, '+553318-1962'],
			[false, '+553616-8064'],
			[false, '+552195-7197'],
			[false, '+553495-3371'],
			[false, '+552189-7520'],
			[false, '+554612-5444'],
			[false, '+552066-8777'],
			[false, '+552314-9837'],
			[false, '+554846-3247'],
			[false, '+553962-5419'],
			[false, '+553784-7659'],
			[false, '+553063-5659'],
			[false, '+552836-4375'],
			[false, '+554094-6965'],
			[false, '+553752-1409'],
			[false, '+554302-8669'],
			[false, '+553357-6994'],
			[false, '+554343-3116'],
			[false, '+552768-4141'],
			[false, '+554352-8339'],
			[false, '+553630-1425'],
			[false, '+552404-9468'],
			[false, '+553294-9192'],
		];
	}
	
	/**
	 * Assert if $random is validated as $expected.
	 *
	 * @covers ::validateRandom
	 * @dataProvider dataRandom
	 * @test Expecting positive assertion.
	 * @param boolean $expected Expected result.
	 * @param boolean $random Random to validate
	 * @return boolean
	 */
	public function canValidateRandom ( bool $expected, string $random )
	{ $this->assertEquals($expected, Parser::validateRandom($random)); }

	/**
	 * A list with random randoms to validate.
	 * Provider to canValidateRandom() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataRandom () : array
	{
		return [
			// Must match dataset
			// UUIDv4
			// Must match dataset
			// UUIDv4
			[true, 'e6b32791-2208-4440-a354-cf554ecc1e13'],
			[true, '64b19041-5436-48b2-8ff3-0b7fe6baad1c'],
			[true, '9295ef01-8978-470c-9fef-09ffa0fc2509'],
			[true, 'b4c95b10-9592-4802-914d-d314d576b8ae'],
			[true, '560d8f3f-4937-4790-bf64-0ea069e7e5c8'],
			[true, '23ae6438-e8f7-4fa2-8daf-95a606fce879'],
			[true, 'c8f8c94d-e088-4bc7-a844-f6fe923587e7'],
			[true, '16875801-cc50-4c4b-9d32-6fc1a8d9c335'],
			[true, '36bb0dd4-58a4-49d2-b9fa-751f79a1f4bf'],
			[true, '1a512ff9-c4a1-4682-a3ef-3ec206ecf3e5'],
			[true, '60327940-21ba-46df-820d-7ce146850959'],
			[true, '3c737964-93eb-4535-b7d3-36fbd13c7d78'],
			[true, 'ec290a2f-8014-40ea-ba4f-b25f7970ba94'],
			[true, 'dffa8609-a5c1-4a1e-83bc-6207c6e037f8'],
			[true, 'f918b30a-bbf1-463c-9315-f27f8c252046'],
			[true, '1a275860-a859-4cd2-ab87-d94a12a025ca'],
			[true, '57fd70cf-905d-4edf-a33e-4ffbbee00c6b'],
			[true, '501c589e-f4df-42b5-98f1-9bd4a1ef8db3'],
			[true, 'b489dea9-3129-49c7-889d-7b06c42d0943'],
			[true, '0dbed350-67e5-48eb-ae81-7d1b74774086'],
			[true, 'e83187e4-e696-46d4-aac9-e4b3a7f7eb5b'],
			[true, 'ad474ad3-b0dc-4f1c-b305-b307fc73cf5e'],
			[true, 'defb58fa-5c5a-4f44-bbd6-9285af57c91f'],
			[true, '031f7b04-84bd-4e38-b01c-2d12bc3132d9'],
			[true, '8e4840b6-ac90-4301-9a03-f691a97296ca'],
			[true, '9ada5557-185e-43e3-ac00-02808173b7fe'],
			[true, 'cdccb93c-2c86-418e-8388-5ac7c6a1e772'],
			[true, 'dba12cdd-4fc4-4f7e-b680-55d6c70f1b89'],
			[true, '5d34c78f-ccb5-4372-a9cc-e590e2a13baa'],
			[true, '82c8d7e3-d300-44a8-b060-e2908145f5a4'],
			[true, '14dacbae-808e-4ef1-b4fc-039cb5e00298'],
			[true, '600f259a-1606-433d-9262-654416bdddd2'],
			[true, 'e860ab7c-0870-4507-a3e0-1199f787f8d8'],
			[true, '0e871790-db08-4f06-bad0-ef8a2a3114f6'],
			[true, '0c7a7750-d6b5-41ff-9378-240dae815c1d'],
			[true, '0da34a35-fd9c-462f-8f72-a22a2aeb3b8d'],
			[true, 'abadf7c0-e075-42ec-893e-eec156b652af'],
			[true, '7a8f2157-c84a-41af-956c-65f6a1a12bdf'],
			[true, '64ee1910-02dc-4fc0-90cf-cc752aec51cd'],
			[true, '5dfe4ecb-9557-44ea-b752-c54a67b2845d'],
			[true, '3c83c998-34b7-405c-bbb1-e623ef80517d'],
			[true, 'e54eb878-9cd3-4628-b269-c7307bbb6146'],
			[true, '061b2ef1-974d-4cf1-bffd-c56ae386a458'],
			[true, '2f3db1da-1764-4861-b6f0-8c8725245391'],
			[true, 'd7c5ff94-e76a-4501-a23b-bf9b93b824b1'],
			[true, '6caa45a9-a109-4975-abd8-7a12a2d70d5d'],
			[true, '3084597e-e7fb-4750-9b29-8bc89fc50ad3'],
			[true, '4148c2cb-8b53-40dc-9240-d8fb137057f2'],
			[true, '8b13e55c-0c95-4434-a544-667bce620c5c'],
			[true, 'fee95221-6b05-4393-a014-d24b5041ccd2'],
			[true, 'b2258677-3f79-4447-814f-dca657004edd'],
			[true, '77e1ed10-2bc7-4370-970a-f8a8af23866e'],
			[true, '3561c23a-667a-4585-83cc-ae25071b2bc6'],
			[true, '74539cf0-4953-4ee3-bb83-bc6655dec585'],
			[true, 'a269cacc-63d8-4d4c-a2e1-2e507ed9093d'],
			[true, '7b6b4c8d-c3e2-466b-8c68-ca4bbd406e00'],
			[true, '412742fc-a32a-487b-90ae-c3bd4b767045'],
			[true, '8154dac2-034d-4fd3-ab82-b5e6d948c88f'],
			[true, '5d1a5361-b951-4fcd-9199-360494544376'],
			[true, 'ddcc1248-379f-4d19-9555-949830287690'],
			[true, '20d1bf9e-e19f-4164-ba74-3039554e46c2'],
			[true, '91299d57-cb31-477e-8422-637e7a6fa388'],
			[true, 'a8627c33-c755-45ce-b864-3ad7cfeedfdb'],
			[true, '6666f92e-e422-40c0-a017-62881de90aa1'],
			[true, '8ce0a0d1-3832-449c-9f38-3b94bca45f18'],
			[true, 'effeadd1-3797-44ac-b4b3-df89e064fbfc'],
			[true, 'd53295ed-4283-48e2-8f96-937f07aae105'],
			[true, '2e4b8436-e2f4-44fc-a031-56360cc713dd'],
			[true, '95b3cea7-e262-40ab-b535-19ec37ed8cd0'],
			[true, '4030c5e1-a0d0-436f-9dfc-c088651b17c5'],
			[true, '0c808630-4380-4b14-be6a-ee6d6ac1f473'],
			[true, '276d660e-a523-4110-ad67-cdc9eaaeefea'],
			[true, 'de08d6df-0d27-4ca5-8e02-36c4fc7a7fdc'],
			[true, 'ea2b1478-cec3-4e8d-999e-415660bfae35'],
			[true, '593860cd-7143-4958-89c9-950dbb4fe96a'],
			[true, '318b0e44-4881-4774-99ef-fa98d7eca071'],
			[true, 'ce58ac70-029d-4031-a44e-bdbe5fc02030'],
			[true, '13f34979-28b0-41e2-9dc1-47a2ffb51bb1'],
			[true, 'd22e0324-cf38-4b85-86eb-8ba36771aa2f'],
			[true, 'bff3458e-18f2-4185-a361-954a259da0c1'],
			[true, '79c69de3-f42d-406c-872a-560caa2763f5'],
			[true, 'c53265fe-4e7d-48b3-a659-17381877b55f'],
			[true, 'ef25868f-3405-4c00-94b1-d1a92dc16cff'],
			[true, 'b8094062-12cd-4b6b-9a74-2d9a245aa985'],
			[true, '3bee217d-2c68-4f42-93a6-5933186952ce'],
			[true, '3f0aada0-3bd4-4404-afe3-bcf5783fd13a'],
			[true, 'eafaf6d0-def8-4f94-ba66-b9396ac0f15b'],
			[true, '8be359ca-e2e8-4928-8529-6d9f076c755f'],
			[true, 'abeb76fd-408d-4b9e-9c3c-27cc2d85d62a'],
			[true, '4e2f2eaa-ac61-4e0c-b999-080523148cf9'],
			[true, 'aeb19e5f-ce5c-4b8b-85e8-68ef686402d8'],
			[true, '72d18c89-cc45-4470-8800-b4849b27c98a'],
			[true, 'a0553631-bfc8-4c98-87e6-cda1c27bb071'],
			[true, '0c046e77-7b0a-4195-8b4e-477773e275f3'],
			[true, 'f123f820-4da8-41ec-9b65-4567d4039818'],
			[true, '7cfc0213-6381-4158-9619-d97ecd71111d'],
			[true, '59f3baef-6931-4496-a0a2-04c6fb9f668e'],
			[true, '819796db-6603-49de-ae6f-db7efb6daf20'],
			[true, '8d3541dc-8603-4413-a99f-fb028c8cd8ec'],
			[true, '49b5eec0-313c-4b4d-ab3e-c8a5d29565d6'],
			// Must not match dataset
			// UUID INVALID
			[false, '79e42b5d-eaa8-34b9-958a-35acb7192a7e'],
			[false, '3caca309-2636-3069-b6fa-88dd8dc3c769'],
			[false, 'eaf123af-a54c-39eb-a087-6a4360fbdc87'],
			[false, 'a19a278f-8ac0-31bc-b8be-d1d00a692a80'],
			[false, '7b2a23a4-e0ed-3e75-95bd-e1d7cc468649'],
			[false, 'c503b64a-4c9c-324a-84b0-52bab86f8690'],
			[false, 'e80c0c30-f144-35cd-8d8a-f6ee2b6586b7'],
			[false, 'c06627ca-7f08-3c17-b95b-6b5992c33028'],
			[false, 'a26e9c79-f527-3f4b-90fc-57dca6d7db4f'],
			[false, '1cb53cc7-c28b-3cb4-9e0b-a7d107c96909'],
			[false, '1501df01-b69a-3646-9d77-8d2b00ef22d5'],
			[false, 'ce61373c-1a32-3ae6-b35e-7df4933f6391'],
			[false, 'f52f7c34-2b84-3fd7-acc4-ea1afcb83786'],
			[false, 'f0aa881d-60eb-325c-beb7-6b2295842cd2'],
			[false, '41490230-f068-3459-bdf9-07cc6b38d8cb'],
			[false, 'a9ba097d-afc2-3e94-bd90-a8086139dbd3'],
			[false, '2108cb92-c845-3691-ae21-517d3c9adb8b'],
			[false, '46abb5dd-b400-3914-8d5f-b4629c6e5da6'],
			[false, '6448ce63-6d96-3e52-b8b0-95e2911fdfb8'],
			[false, 'ca5c66ca-a437-392f-910c-867b70b3739f'],
			[false, 'f19e77bb-ad20-372b-a124-5f2ad34cde00'],
			[false, '215e1077-b1cc-37c7-bec4-6908864ba7ec'],
			[false, 'c54e52d4-6c66-3470-8274-fb6561f8a22b'],
			[false, 'b0357e91-0cb5-3208-89ae-e44d4b48b67e'],
			[false, '2f68aba3-025b-3a3d-9cc2-d5c43679f454'],
			[false, '7b806a6d-17c8-3579-b141-f51bd517b5ae'],
			[false, 'ddeb8d97-e840-32d0-8b5a-bd263fbc41f2'],
			[false, '5f170b02-a042-3fc4-8ef3-37ac10231d9c'],
			[false, '9c0aaa93-c8f7-3e20-afc7-6ea1ae409dc3'],
			[false, '8e4339d9-8206-372a-95e8-f52b0f8e4dae'],
			[false, '28019b8d-5e01-3920-99a7-263c0faf0311'],
			[false, '2df1552f-1ff2-3dcd-8018-c6f49a6536c9'],
			[false, '0c9fb1bb-f39e-3950-a921-4b4779fa4f67'],
			[false, '46b60592-f7d5-3053-80b9-ce591639d7ad'],
			[false, '50f5fddf-4e9d-3128-9415-fc08e1778788'],
			[false, 'f9847be2-09d8-3ba3-ba89-fb8ba28f1a64'],
			[false, '7794577c-38b9-3d21-bc88-5f74743913d1'],
			[false, '08a0d2bb-3672-3719-8ef8-36cd255b2c8f'],
			[false, 'd488aed6-0966-383c-ae91-d46b017cefef'],
			[false, 'c426691e-14b8-3401-b98c-1b57cc1ff7f2'],
			[false, '69122f54-271f-3f1e-b18d-4207df97b67e'],
			[false, '65da2647-e6f5-3cc4-8d24-df5bd58c857b'],
			[false, '7606c1d5-7144-3bcd-86d9-aac29f110e58'],
			[false, 'ad60ab6a-48b5-33d2-8386-e46df3a48926'],
			[false, '9c89a593-a3c7-3466-987b-db6c5a0e2c80'],
			[false, '3b75e029-d123-34cb-b2c8-10550ac68a14'],
			[false, '5b566b4f-8daa-3058-8fe9-f75191223eb2'],
			[false, '1685779c-5a8a-3f37-8d06-aa004c307f72'],
			[false, '12d9a382-0d7d-3797-8aba-e571e78ed43e'],
			[false, 'f303cbc5-4d85-3bc1-bc4e-7b2d14779e0e'],
		];
	}

	/**
	 * Assert if $data is the $expected key type.
	 *
	 * @covers ::parseDocument
	 * @covers ::getKeyType
	 * @dataProvider dataKeyType
	 * @test Expecting positive assertion.
	 * @param string $expected Expected result.
	 * @param string $data KeyType to validate
	 * @return boolean
	 */
	public function canGetKeyType ( string $expected, string $data )
	{ $this->assertEquals($expected, Parser::getKeyType($data)); }

	/**
	 * A list with random data to validate.
	 * Provider to canGetKeyType() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataKeyType () : array
	{
		return [
			// Must match dataset
			['document', '93907932000101'],
			['document', '85260231000197'],
			['document', '37257127000109'],
			['document', '14699979000142'],
			['document', '50636572000130'],
			['document', '57703415000149'],
			['document', '37312334000100'],
			['document', '67781770000190'],
			['document', '82889624000176'],
			['document', '10651651000103'],
			['document', '39433829000113'],
			['document', '93715041000145'],
			['document', '59745148000106'],
			['document', '30722684000104'],
			['document', '37300479000191'],
			['document', '96276739000108'],
			['document', '65789992000123'],
			['document', '79091604000140'],
			['document', '71276040000109'],
			['document', '74332001000170'],
			['document', '77981254000162'],
			['document', '56822848000150'],
			['document', '43694975000170'],
			['document', '80354760000136'],
			['document', '95550599000143'],
			['document', '30104708000153'],
			['document', '96165558000104'],
			['document', '11176777000128'],
			['document', '64986014000109'],
			['document', '93847754000162'],
			['document', '23645508000142'],
			['document', '56286337000161'],
			['document', '40375127000100'],
			['document', '20687554000116'],
			['document', '07509721000170'],
			['document', '97476375000164'],
			['document', '09471761000104'],
			['document', '93914315000125'],
			['document', '68742347000143'],
			['document', '59044217000146'],
			['document', '18702463000160'],
			['document', '59644520000180'],
			['document', '20241297000194'],
			['document', '81814930000180'],
			['document', '68181089000173'],
			['document', '10697488000101'],
			['document', '71304309000113'],
			['document', '73631275000105'],
			['document', '16548737000183'],
			['document', '60609737000138'],
			['document', '13828718647'],
			['document', '56631648203'],
			['document', '08427591152'],
			['document', '30130651842'],
			['document', '04842708425'],
			['document', '89553202217'],
			['document', '14079544820'],
			['document', '17413546418'],
			['document', '05926776481'],
			['document', '65456915626'],
			['document', '50900305452'],
			['document', '80978125843'],
			['document', '55180644623'],
			['document', '09699598778'],
			['document', '05572167614'],
			['document', '09431568667'],
			['document', '85010220555'],
			['document', '87058722196'],
			['document', '15447619530'],
			['document', '33541026081'],
			['document', '07721756910'],
			['document', '14967025811'],
			['document', '75991071500'],
			['document', '29399624480'],
			['document', '96529761489'],
			['document', '01997014661'],
			['document', '57008142340'],
			['document', '33224342730'],
			['document', '10183559959'],
			['document', '18421774948'],
			['document', '32254791010'],
			['document', '27070863713'],
			['document', '06470879365'],
			['document', '57149720072'],
			['document', '11202995861'],
			['document', '69026443145'],
			['document', '27412671545'],
			['document', '00006707823'],
			['document', '03419281005'],
			['document', '95917779813'],
			['document', '75908095605'],
			['document', '05530328873'],
			['document', '64862357644'],
			['document', '57821942359'],
			['document', '16637248142'],
			['document', '72566713006'],
			['document', '31838229035'],
			['document', '15311234571'],
			['document', '00027214818'],
			['document', '78783478752'],
			['email', 'epadilha@uol.com.br'],
			['email', 'adriano.galindo@velasques.com.br'],
			['email', 'eduardo77@terra.com.br'],
			['email', 'sepulveda.yohanna@uol.com.br'],
			['email', 'naiara.uchoa@teles.com.br'],
			['email', 'gilberto.chaves@r7.com'],
			['email', 'xleon@gmail.com'],
			['email', 'medina.aline@escobar.com'],
			['email', 'delvalle.joyce@uol.com.br'],
			['email', 'lbeltrao@bittencourt.net.br'],
			['email', 'olga.galvao@sanches.com.br'],
			['email', 'dearruda.talita@camacho.com.br'],
			['email', 'lorena48@serrano.org'],
			['email', 'mortega@martines.com'],
			['email', 'leo00@hotmail.com'],
			['email', 'marcos84@r7.com'],
			['email', 'guilherme49@sales.com'],
			['email', 'flavio.faro@hotmail.com'],
			['email', 'qbranco@uol.com.br'],
			['email', 'fontes.franco@yahoo.com'],
			['email', 'sandra.ferraz@galindo.com'],
			['email', 'icortes@ig.com.br'],
			['email', 'carla41@r7.com'],
			['email', 'paes.augusto@ig.com.br'],
			['email', 'daniella00@madeira.org'],
			['email', 'gdeverso@leon.net'],
			['email', 'ovalentin@lovato.com.br'],
			['email', 'rvieira@hotmail.com'],
			['email', 'sara.arruda@burgos.com'],
			['email', 'deverso.hugo@colaco.net'],
			['email', 'carla.roque@terra.com.br'],
			['email', 'gleal@hotmail.com'],
			['email', 'isabel.rocha@maldonado.com.br'],
			['email', 'luara60@cortes.br'],
			['email', 'rezende.miguel@terra.com.br'],
			['email', 'vdasdores@bonilha.net.br'],
			['email', 'nathalia00@jimenes.com.br'],
			['email', 'ramires.tais@hotmail.com'],
			['email', 'cordeiro.dante@terra.com.br'],
			['email', 'vjimenes@queiros.com.br'],
			['email', 'rafael95@paz.net.br'],
			['email', 'cintia05@zambrano.br'],
			['email', 'bruna.caldeira@ig.com.br'],
			['email', 'beltrao.kesia@esteves.net.br'],
			['email', 'mariana.amaral@chaves.br'],
			['email', 'amaral.mari@gmail.com'],
			['email', 'pacheco.thais@r7.com'],
			['email', 'serrano.samanta@terra.com.br'],
			['email', 'julio.vila@yahoo.com'],
			['email', 'veronica.montenegro@galhardo.com'],
			['email', 'sergio.mares dias.com'],
			['email', 'hfeliciano ramires.br'],
			['email', 'christopher80 ig.com.br'],
			['email', 'catarina.carmona campos.org'],
			['email', 'yrosa gmail.com'],
			['email', 'kauan47 terra.com.br'],
			['email', 'wmontenegro galvao.com.br'],
			['email', 'bdelvalle r7.com'],
			['email', 'sabrina66 leon.org'],
			['email', 'malves serna.com'],
			['email', 'qtamoio terra.com.br'],
			['email', 'edelvalle r7.com'],
			['email', 'asalas bittencourt.net.br'],
			['email', 'nathalia.dasneves faro.net'],
			['email', 'carmona.mary verdugo.com'],
			['email', 'hdacruz terra.com.br'],
			['email', 'dominato.saulo yahoo.com'],
			['email', 'paz.maiara yahoo.com'],
			['email', 'enzo.maldonado terra.com.br'],
			['email', 'daniella.gomes deverso.com'],
			['email', 'cecilia.toledo r7.com'],
			['email', 'stella.ferreira bonilha.net.br'],
			['email', 'michelle.pena uol.com.br'],
			['email', 'fernando.uchoa terra.com.br'],
			['email', 'fernandes.sabrina yahoo.com'],
			['email', 'hpedrosa meireles.net'],
			['email', 'delgado.ketlin hotmail.com'],
			['email', 'caio88 r7.com'],
			['email', 'julieta.sales prado.com.br'],
			['email', 'estrada.guilherme ramos.net.br'],
			['email', 'danilo.galindo benez.com'],
			['email', 'janaina.lutero duarte.net.br'],
			['email', 'joana61 rosa.br'],
			['email', 'hugo78 galindo.net.br'],
			['email', 'fatima.chaves yahoo.com'],
			['email', 'ingrid36 valente.com.br'],
			['email', 'ideoliveira r7.com'],
			['email', 'serrano.debora delgado.org'],
			['email', 'verdugo.pedro r7.com'],
			['email', 'manuela.ferreira ortiz.org'],
			['email', 'zmarin ig.com.br'],
			['email', 'sales.paloma aguiar.net.br'],
			['email', 'james.reis uol.com.br'],
			['email', 'gian45 faria.com'],
			['email', 'michele.velasques terra.com.br'],
			['email', 'rodrigues.ester teles.net'],
			['email', 'mlira uol.com.br'],
			['email', 'brito.eva r7.com'],
			['email', 'josefina64 campos.net'],
			['email', 'vila.estevao terra.com.br'],
			['phone', '+558629428099'],
			['phone', '+5554905360168'],
			['phone', '+5564945138781'],
			['phone', '+5549993933802'],
			['phone', '+551331913817'],
			['phone', '+5566937856173'],
			['phone', '+558926603747'],
			['phone', '+5545988539404'],
			['phone', '+5541938304256'],
			['phone', '+557941859553'],
			['phone', '+558221116389'],
			['phone', '+5546952033293'],
			['phone', '+5551955139504'],
			['phone', '+555426060032'],
			['phone', '+558641487805'],
			['phone', '+554446070332'],
			['phone', '+5565940493558'],
			['phone', '+5579920493534'],
			['phone', '+555138705587'],
			['phone', '+5533917478848'],
			['phone', '+5515949386703'],
			['phone', '+552420173444'],
			['phone', '+553845276408'],
			['phone', '+551330910538'],
			['phone', '+556120425477'],
			['phone', '+554137837419'],
			['phone', '+5512949284686'],
			['phone', '+5528911556203'],
			['phone', '+551244003561'],
			['phone', '+553142835824'],
			['phone', '+5575914969759'],
			['phone', '+5586912502180'],
			['phone', '+5575918100703'],
			['phone', '+559733103649'],
			['phone', '+551439317283'],
			['phone', '+5594997225256'],
			['phone', '+5512958299810'],
			['phone', '+556331005197'],
			['phone', '+5522966972828'],
			['phone', '+557545919875'],
			['phone', '+554223345589'],
			['phone', '+556846756123'],
			['phone', '+5521946575390'],
			['phone', '+5551962671490'],
			['phone', '+5573906164034'],
			['phone', '+551247735039'],
			['phone', '+553222645703'],
			['phone', '+5515906782414'],
			['phone', '+552446101687'],
			['phone', '+5582988993075'],
			['random', 'bb25712a-4202-496a-9691-ebb54cc962d4'],
			['random', '05231a6b-9580-4fe4-9134-cedc1d851921'],
			['random', 'b7c4e43b-9b9f-46c2-b625-71822f9b6636'],
			['random', 'cc4667dd-6fc8-46ef-8906-a9f0978992e7'],
			['random', 'ee08796e-9d5b-47f1-ab9e-a178ed80ba76'],
			['random', '193f22cf-e262-40da-a633-afe926ffaeb5'],
			['random', 'c386c2ee-4f94-4a4e-af93-3fe1bba1297b'],
			['random', '81b38cd0-4db5-4a27-af1f-c795e5290668'],
			['random', 'a44be048-cbea-4ca8-afcb-634268c3ae2b'],
			['random', '4fd58e8f-785c-4c20-8233-378a224a3b82'],
			['random', '79cb18eb-e3a6-440c-9da1-144bd530a5ad'],
			['random', '3a630b7d-dd14-4ac6-9b0e-299756528fb0'],
			['random', 'fcfa8481-86ce-4f20-8576-d4fe8902a01e'],
			['random', '4c15502b-88ce-4770-b1ba-2c8b2d883147'],
			['random', 'e00818cd-df7f-4891-bac7-7fdba4fbbfe3'],
			['random', 'be171072-62d8-4442-898b-d21645e69955'],
			['random', '099d42f3-349d-4f25-be7a-c5b41c491582'],
			['random', '2a0ab805-2983-4471-82e3-611c2b12861f'],
			['random', '7bc29461-4259-48ed-8708-2699b354e83e'],
			['random', '6fe4da4d-c432-4990-9b7a-a4d38a8ef111'],
			['random', '0fb85a56-5a21-4a24-a91f-b0e0c5f2b1b5'],
			['random', 'dcc23033-8708-4f41-99ca-d25d04dd3eb8'],
			['random', '9a78ce4b-2bdc-4673-b1dd-3346cd17fab2'],
			['random', '1d0ba1f7-7f81-42ee-9480-c8f630317753'],
			['random', '499eb908-f7d5-4aa6-880a-db0fc2b23523'],
			['random', '129b0215-afbd-4291-9914-61f3c428e04a'],
			['random', '2ad95e67-9653-4aaf-aeb3-c00c9686e163'],
			['random', '4d61840e-1bbc-4d4e-8f03-1077f868d32b'],
			['random', 'dd81acba-7b51-4a98-b683-544ccf86d219'],
			['random', 'b43b8759-2f65-4a6f-9621-62535a80f16e'],
			['random', 'd7610d5b-bb92-4bb4-a5ea-8f7f371002d4'],
			['random', '6054009c-a599-4822-a299-213846ee8b67'],
			['random', '5c762b32-e832-4527-bf5f-6a6d44dd9577'],
			['random', '84ea1ee0-d3e5-4b9c-8929-3d6b18af17e6'],
			['random', 'da8990ab-ed6d-44ac-9f6a-20f004879630'],
			['random', 'de224c17-9c28-4f59-beee-f16607cb0a59'],
			['random', '39e138de-c4d1-4c5a-85cb-b2306c948879'],
			['random', '8fa81060-8845-420c-8cc7-c3e956deacf9'],
			['random', '4da7fca5-1239-448a-b0f3-faf97e7df329'],
			['random', 'd7d04ed5-6201-4359-adcb-768ef1952891'],
			['random', '9f51a7c6-ab07-49a9-8376-f08ee29a20b5'],
			['random', '8c2430bb-3a81-4a3d-9b6b-ee4ecbd86cd3'],
			['random', '0b7c8428-c26e-40e8-9503-b932fa0c4e9f'],
			['random', '8e378405-e035-4b54-a264-d1f39cec45dc'],
			['random', 'b621415b-9e44-42eb-a587-5e043cc3798d'],
			['random', '6da68c24-8143-4282-b6bd-1b3687e9161a'],
			['random', '7c0e1784-0d9b-40e8-946d-739250d14b22'],
			['random', '445d73be-2984-4e6b-a656-46c6ac293f3c'],
			['random', 'f6938484-217f-485b-9a74-aa32da381c7d'],
			['random', 'd4ad12a4-cdf8-4e9d-be2a-61e2ea21bd75'],
		];
	}
	

	/**
	 * Assert if $data is the $expected key type.
	 *
	 * @covers ::parseDocument
	 * @covers ::getKeyType
	 * @dataProvider dataInvalidKeyType
	 * @test Expecting positive assertion.
	 * @param string $data KeyType to validate
	 * @return boolean
	 */
	public function cannotGetKeyType ( string $data )
	{ 
		$this->expectException(Exception::class);
		Parser::getKeyType($data);
	}

	/**
	 * A list with random data to validate.
	 * Provider to cannotGetKeyType() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataInvalidKeyType () : array
	{
		return [
			// Must not match dataset
			['38182315447946'],
			['78039245422995'],
			['69305080311632'],
			['26512900228462'],
			['48047098017608'],
			['37639775727293'],
			['16475521201365'],
			['09596547179279'],
			['50881895638544'],
			['78776990605994'],
			['46922194955199'],
			['26270641051909'],
			['00165978485419'],
			['73372166473131'],
			['30594636532950'],
			['57633131009292'],
			['14846429308071'],
			['46158879246327'],
			['34379177436853'],
			['44216807094900'],
			['36695729580231'],
			['97754131084320'],
			['78802056941071'],
			['53241389588732'],
			['13545160392163'],
			['69172452364149'],
			['00697008007871'],
			['09728185224089'],
			['18893885708965'],
			['48368655446555'],
			['14660833007879'],
			['39466545000997'],
			['73838257278890'],
			['25735666938192'],
			['42130072807956'],
			['59272655112204'],
			['19804505808764'],
			['51062608620212'],
			['24750795475252'],
			['39289191231716'],
			['56324767249213'],
			['95926129950578'],
			['40536842788214'],
			['26619040053762'],
			['53925701823729'],
			['97746084211222'],
			['34369248613463'],
			['08366273922655'],
			['71198968426424'],
			['05945649783150'],
			['431672877'],
			['702748990'],
			['234989957'],
			['878517377'],
			['129153626'],
			['350692924'],
			['524523554'],
			['645884172'],
			['743321369'],
			['474658884'],
			['923592560'],
			['826080149'],
			['634141907'],
			['083747509'],
			['944288581'],
			['151771137'],
			['654652913'],
			['541919685'],
			['007272976'],
			['835380920'],
			['223178029'],
			['079624109'],
			['095068062'],
			['021370651'],
			['853617347'],
			['002652520'],
			['990977417'],
			['816280711'],
			['915714344'],
			['951694758'],
			['485082234'],
			['301790308'],
			['105966988'],
			['183011613'],
			['789587481'],
			['334367933'],
			['764310969'],
			['260096936'],
			['856496022'],
			['584611689'],
			['407928703'],
			['910903071'],
			['258495447'],
			['716017075'],
			['937518974'],
			['180452315'],
			['885830760'],
			['510721083'],
			['351067106'],
			['515914682'],
			['amaral.veronicar7.com'],
			['adearrudarocha.com.br'],
			['italo.pontesgmail.com'],
			['madalena76rios.com'],
			['guilherme.roquer7.com'],
			['priscila65darosa.com.br'],
			['mbritomascarenhas.com.br'],
			['aragao.leonardoig.com.br'],
			['lidiane.delvalleuol.com.br'],
			['dasneves.fabriciosantacruz.com'],
			['pedrosa.agostinhoig.com.br'],
			['hqueirosfontes.com'],
			['lavinia.lourencofonseca.com'],
			['bittencourt.heloiseazevedo.com.br'],
			['avila.giseler7.com'],
			['quintana.carolpaz.org'],
			['esteves.anahotmail.com'],
			['nicole.coronaoliveira.net'],
			['priscila68terra.com.br'],
			['benedito.pontesserrano.org'],
			['tatiane.deversoserna.net.br'],
			['gpereiraaranda.com.br'],
			['ester.molinahotmail.com'],
			['assuncao.barbarahotmail.com'],
			['carvalho.giovanehotmail.com'],
			['mirella08hotmail.com'],
			['stephany.ferreiradelvalle.com.br'],
			['leia74yahoo.com'],
			['sophia.jimenesgalindo.br'],
			['tessalia.chavesdacruz.com'],
			['benites.victorespinoza.br'],
			['solano.renandavila.org'],
			['kesia.ramiresbeltrao.com'],
			['jrezendeyahoo.com'],
			['andres.luteror7.com'],
			['jose.rodriguesmarin.com'],
			['juliane.ferreiraterra.com.br'],
			['osernayahoo.com'],
			['catarina87yahoo.com'],
			['isabel.matoszamana.net'],
			['alessandro.correiauol.com.br'],
			['valentin.padraouol.com.br'],
			['rivera.samarahotmail.com'],
			['stefany19uol.com.br'],
			['karina.zambranoyahoo.com'],
			['bianca65ig.com.br'],
			['pereira.brunayahoo.com'],
			['dbrancozambrano.com'],
			['serra.hortenciaburgos.br'],
			['joao06terra.com.br'],
			['9bfaaf48-9dbb-38e4-b58d-151e88550eeb'],
			['f62bc9f8-6877-374e-a4bc-abb3a49a0674'],
			['ec1b8c79-8f78-3e7b-882c-2323c285fc23'],
			['639f7037-0f25-38f7-a886-657c80a6f1d7'],
			['d7373f8c-0b09-378a-b166-c5c4cac0e6e0'],
			['501f0743-5e02-3784-94e3-0b03fc523bef'],
			['c7bbeecc-b81a-375e-ba2d-ae3b8550f11e'],
			['82de7ac4-acc9-38f1-815a-bee9e8fce2ac'],
			['197d350b-a148-303e-b1da-6e2295e35b49'],
			['11cfacb5-5a3f-3ce2-8548-3ad26d21213d'],
			['6a149eb7-8587-3138-a2b5-c571118849b3'],
			['42b8edad-ed6a-32b9-a97a-2cac0a12d85b'],
			['22e94375-fa2f-3b92-bb68-d7220ef158d0'],
			['b8faab07-0417-33b9-85a6-59aca486688f'],
			['c309c0b7-a0be-312e-944d-7e4ea7a922d9'],
			['285df39a-d5d7-3dc0-b855-75f78367ce9c'],
			['9adc8e16-d219-30b3-9ea5-702af3e7b3c1'],
			['b16aef64-82e0-3dd8-b1c4-1047974c38ef'],
			['159400d3-7a82-3b05-b726-e273bf9b748e'],
			['8d0115fd-c31f-3d90-8c47-96aca19460d1'],
			['4803d6ec-e97f-3e9d-8847-9770794c56a8'],
			['6e8f8f1e-8d7a-38ae-85dd-03f2df7b90f2'],
			['fb397fd4-d46c-334a-b313-a4f1249c669e'],
			['962a11c5-0d94-31ab-b299-0809f2c94abb'],
			['b3a738df-6d6c-3953-9a80-71d24ea11b73'],
			['ee7284a4-dbc2-38d7-b413-fa23b9e6539f'],
			['e000c60d-5168-3093-ba96-9873414bd5bd'],
			['ff3b43a4-4a9e-3d81-8711-75ce445d36b8'],
			['f48c9752-5f10-349d-876e-3935cda74a11'],
			['ba0719ed-5e7e-3c2c-9165-192ac23bdefd'],
			['a7bbc052-7ff7-39f7-a2a3-9ae59cfeed3c'],
			['e478f30e-b5ea-3eab-8cc5-cf5c398bd9af'],
			['f841e8d2-0602-399d-8231-b97e4a3a49c6'],
			['a5d0eede-62cb-3b26-a5d7-d16e861bb694'],
			['a1a1a5b1-870b-313f-addc-c5693b2ba3da'],
			['2048ac45-919d-3387-9feb-fdde78705149'],
			['157208ce-8b6c-30da-a7a5-dc78d6b659a5'],
			['ae04a8ab-51ba-3165-a5cf-1b0db0a4fca0'],
			['2cddc1e8-ddbf-34f5-9d2e-453432c31964'],
			['7781b621-39a2-3d0e-a8fd-245a116bf2dc'],
			['093859ce-65f0-3e5f-b007-257fb16f4f76'],
			['54dd4648-37b4-352f-9fa3-235eecb7335f'],
			['6f247ee4-b350-3cec-83fc-513992636011'],
			['2b87b28c-ee20-31fd-a04e-0d1bf3f9eec4'],
			['d99c8a87-8b7b-3f96-88a8-3553f809f214'],
			['48d620f7-7d49-3c89-a8eb-c69b9930e5b6'],
			['1264f403-4f81-3fc3-af41-5a383885d05a'],
			['16c6c79a-601a-36dc-baf6-a0ae0d21f7ae'],
			['f21b90f8-e6ff-38ba-957f-45828ddc3754'],
			['191962a4-c20b-3fdf-98da-0e8370e37b65'],
		];
	}
}